"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.mimeTypeSymbol = exports.freezeSymbol = exports.filesSymbol = void 0;
const tslib_1 = require("tslib");
const node_stream_1 = require("node:stream");
const node_buffer_1 = require("node:buffer");
const do_not_zip_1 = require("do-not-zip");
const Messages = tslib_1.__importStar(require("./messages.js"));
exports.filesSymbol = Symbol("bundleFiles");
exports.freezeSymbol = Symbol("bundleFreeze");
exports.mimeTypeSymbol = Symbol("bundleMimeType");
/**
 * Defines a container ready to be distributed.
 * If no mimeType is passed to the constructor,
 * it will throw an error.
 */
class Bundle {
    constructor(mimeType) {
        this[_a] = {};
        if (!mimeType) {
            throw new Error(Messages.BUNDLE.MIME_TYPE_MISSING);
        }
        this[exports.mimeTypeSymbol] = mimeType;
    }
    /**
     * Creates a bundle and exposes the
     * function to freeze it manually once
     * completed.
     *
     * This was made to not expose freeze
     * function outside of Bundle class.
     *
     * Normally, a bundle would get freezed
     * when using getAsBuffer or getAsStream
     * but when creating a PKPasses archive,
     * we need to freeze the bundle so the
     * user cannot add more files (we want to
     * allow them to only the selected files)
     * but also letting them choose how to
     * export it.
     *
     * @param mimeType
     * @returns
     */
    static freezable(mimeType) {
        const bundle = new Bundle(mimeType);
        return [bundle, () => bundle[exports.freezeSymbol]()];
    }
    /**
     * Retrieves bundle's mimeType
     */
    get mimeType() {
        return this[exports.mimeTypeSymbol];
    }
    /**
     * Freezes the bundle so no more files
     * can be added any further.
     */
    [(_a = exports.filesSymbol, exports.freezeSymbol)]() {
        if (this.isFrozen) {
            return;
        }
        Object.freeze(this[exports.filesSymbol]);
    }
    /**
     * Tells if this bundle still allows files to be added.
     * @returns false if files are allowed, true otherwise
     */
    get isFrozen() {
        return Object.isFrozen(this[exports.filesSymbol]);
    }
    /**
     * Returns a copy of the current list of buffers
     * that have been added to the class.
     *
     * It does not include translation files, manifest
     * and signature.
     *
     * Final files list might differ due to export
     * conditions.
     */
    get files() {
        return Object.keys(this[exports.filesSymbol]);
    }
    /**
     * Allows files to be added to the bundle.
     * If the bundle is closed, it will throw an error.
     *
     * @param fileName
     * @param buffer
     */
    addBuffer(fileName, buffer) {
        if (this.isFrozen) {
            throw new Error(Messages.BUNDLE.CLOSED);
        }
        this[exports.filesSymbol][fileName] = buffer;
    }
    /**
     * Closes the bundle and returns it as a Buffer.
     * Once closed, the bundle does not allow files
     * to be added any further.
     *
     * @returns Buffer
     */
    getAsBuffer() {
        this[exports.freezeSymbol]();
        return node_buffer_1.Buffer.from((0, do_not_zip_1.toArray)(createZipFilesMap(this[exports.filesSymbol])));
    }
    /**
     * Closes the bundle and returns it as a stream.
     * Once closed, the bundle does not allow files
     * to be added any further.
     *
     * @returns
     */
    getAsStream() {
        this[exports.freezeSymbol]();
        return node_stream_1.Readable.from(node_buffer_1.Buffer.from((0, do_not_zip_1.toArray)(createZipFilesMap(this[exports.filesSymbol]))));
    }
    /**
     * Closes the bundle and returns it as an object.
     * This allows developers to choose a different way
     * of serving, analyzing or zipping the file, outside the
     * default compression system.
     *
     * @returns a frozen object containing files paths as key
     * 		and Buffers as content.
     */
    getAsRaw() {
        this[exports.freezeSymbol]();
        return Object.freeze({ ...this[exports.filesSymbol] });
    }
}
exports.default = Bundle;
/**
 * Creates a files map for do-not-zip
 *
 * @param files
 * @returns
 */
function createZipFilesMap(files) {
    return Object.entries(files).map(([path, data]) => ({
        path,
        data,
    }));
}
//# sourceMappingURL=data:application/json;base64,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