"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const Schemas = tslib_1.__importStar(require("./schemas/index.js"));
const Utils = tslib_1.__importStar(require("./utils.js"));
const Messages = tslib_1.__importStar(require("./messages.js"));
/**
 * Class to represent lower-level keys pass fields
 * @see https://apple.co/2wkUBdh
 */
const passInstanceSymbol = Symbol("passInstance");
const sharedKeysPoolSymbol = Symbol("keysPool");
const fieldSchemaSymbol = Symbol("fieldSchema");
class FieldsArray extends Array {
    constructor(passInstance, keysPool, fieldSchema, ...args) {
        super(...args);
        this[fieldSchemaSymbol] = fieldSchema;
        this[passInstanceSymbol] = passInstance;
        this[sharedKeysPoolSymbol] = keysPool;
    }
    push(...items) {
        const validItems = registerWithValidation(this, ...items);
        return super.push(...validItems);
    }
    pop() {
        return unregisterItems(this, () => super.pop());
    }
    splice(start, deleteCount, ...items) {
        // Perfoming frozen check, validation and getting valid items
        const validItems = registerWithValidation(this, ...items);
        for (let i = start; i < start + deleteCount; i++) {
            this[sharedKeysPoolSymbol].delete(this[i].key);
        }
        return super.splice(start, deleteCount, ...validItems);
    }
    shift() {
        return unregisterItems(this, () => super.shift());
    }
    unshift(...items) {
        const validItems = registerWithValidation(this, ...items);
        return super.unshift(...validItems);
    }
}
exports.default = FieldsArray;
function registerWithValidation(instance, ...items) {
    Utils.assertUnfrozen(instance[passInstanceSymbol]);
    let validItems = [];
    for (const field of items) {
        if (!field) {
            console.warn(Messages.format(Messages.FIELDS.INVALID, field));
            continue;
        }
        try {
            Schemas.assertValidity(instance[fieldSchemaSymbol], field, Messages.FIELDS.INVALID);
            if (instance[sharedKeysPoolSymbol].has(field.key)) {
                throw new TypeError(Messages.format(Messages.FIELDS.REPEATED_KEY, field.key));
            }
            instance[sharedKeysPoolSymbol].add(field.key);
            validItems.push(field);
        }
        catch (err) {
            if (err instanceof Error) {
                console.warn(err.message ? err.message : err);
            }
            else {
                console.warn(err);
            }
        }
    }
    return validItems;
}
function unregisterItems(instance, removeFn) {
    Utils.assertUnfrozen(instance[passInstanceSymbol]);
    const element = removeFn();
    instance[sharedKeysPoolSymbol].delete(element.key);
    return element;
}
//# sourceMappingURL=data:application/json;base64,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