"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.default = getModelFolderContents;
const tslib_1 = require("tslib");
const path = tslib_1.__importStar(require("node:path"));
const node_fs_1 = require("node:fs");
const Utils = tslib_1.__importStar(require("./utils.js"));
const Messages = tslib_1.__importStar(require("./messages.js"));
/**
 * Reads the model folder contents
 *
 * @param model
 * @returns A promise of an object containing all
 * 		filePaths and the relative buffer
 */
async function getModelFolderContents(model) {
    try {
        const modelPath = `${model}${(!path.extname(model) && ".pass") || ""}`;
        const modelFilesList = await node_fs_1.promises.readdir(modelPath);
        // No dot-starting files, manifest and signature and only files with an extension
        const modelSuitableRootPaths = Utils.removeHidden(modelFilesList).filter((f) => !/(manifest|signature)/i.test(f) &&
            /.+$/.test(path.parse(f).ext));
        const modelRecords = await Promise.all(modelSuitableRootPaths.map((fileOrDirectoryPath) => readFileOrDirectory(path.resolve(modelPath, fileOrDirectoryPath))));
        return Object.fromEntries(modelRecords.flat(1));
    }
    catch (err) {
        if (!isErrorErrNoException(err) || !isMissingFileError(err)) {
            throw err;
        }
        if (isFileReadingFailure(err)) {
            throw new Error(Messages.format(Messages.MODELS.FILE_NO_OPEN, JSON.stringify(err)));
        }
        if (isDirectoryReadingFailure(err)) {
            throw new Error(Messages.format(Messages.MODELS.DIR_NOT_FOUND, err.path));
        }
        throw err;
    }
}
function isErrorErrNoException(err) {
    return Object.prototype.hasOwnProperty.call(err, "errno");
}
function isMissingFileError(err) {
    return err.code === "ENOENT";
}
function isDirectoryReadingFailure(err) {
    return err.syscall === "scandir";
}
function isFileReadingFailure(err) {
    return err.syscall === "open";
}
/**
 * Allows reading both a whole directory or a set of
 * file in the same flow
 *
 * @param filePath
 * @returns
 */
async function readFileOrDirectory(filePath) {
    const stats = await node_fs_1.promises.lstat(filePath);
    if (stats.isDirectory()) {
        return readFilesInDirectory(filePath);
    }
    else {
        return getFileContents(filePath).then((result) => [result]);
    }
}
/**
 * Reads a directory and returns all
 * the files in it
 *
 * @param filePath
 * @returns
 */
async function readFilesInDirectory(filePath) {
    const dirContent = await node_fs_1.promises.readdir(filePath).then(Utils.removeHidden);
    return Promise.all(dirContent.map((fileName) => getFileContents(path.resolve(filePath, fileName), 2)));
}
/**
 * @param filePath
 * @param pathSlicesDepthFromEnd used to preserve localization lproj content
 * @returns
 */
async function getFileContents(filePath, pathSlicesDepthFromEnd = 1) {
    const fileComponents = filePath.split(path.sep);
    const fileName = fileComponents
        .slice(fileComponents.length - pathSlicesDepthFromEnd)
        .join("/");
    const content = await node_fs_1.promises.readFile(filePath);
    return [fileName, content];
}
//# sourceMappingURL=data:application/json;base64,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