var _a;
import { Readable } from "node:stream";
import { Buffer } from "node:buffer";
import { toArray as zipToArray } from "do-not-zip";
import * as Messages from "./messages.js";
export const filesSymbol = Symbol("bundleFiles");
export const freezeSymbol = Symbol("bundleFreeze");
export const mimeTypeSymbol = Symbol("bundleMimeType");
/**
 * Defines a container ready to be distributed.
 * If no mimeType is passed to the constructor,
 * it will throw an error.
 */
export default class Bundle {
    constructor(mimeType) {
        this[_a] = {};
        if (!mimeType) {
            throw new Error(Messages.BUNDLE.MIME_TYPE_MISSING);
        }
        this[mimeTypeSymbol] = mimeType;
    }
    /**
     * Creates a bundle and exposes the
     * function to freeze it manually once
     * completed.
     *
     * This was made to not expose freeze
     * function outside of Bundle class.
     *
     * Normally, a bundle would get freezed
     * when using getAsBuffer or getAsStream
     * but when creating a PKPasses archive,
     * we need to freeze the bundle so the
     * user cannot add more files (we want to
     * allow them to only the selected files)
     * but also letting them choose how to
     * export it.
     *
     * @param mimeType
     * @returns
     */
    static freezable(mimeType) {
        const bundle = new Bundle(mimeType);
        return [bundle, () => bundle[freezeSymbol]()];
    }
    /**
     * Retrieves bundle's mimeType
     */
    get mimeType() {
        return this[mimeTypeSymbol];
    }
    /**
     * Freezes the bundle so no more files
     * can be added any further.
     */
    [(_a = filesSymbol, freezeSymbol)]() {
        if (this.isFrozen) {
            return;
        }
        Object.freeze(this[filesSymbol]);
    }
    /**
     * Tells if this bundle still allows files to be added.
     * @returns false if files are allowed, true otherwise
     */
    get isFrozen() {
        return Object.isFrozen(this[filesSymbol]);
    }
    /**
     * Returns a copy of the current list of buffers
     * that have been added to the class.
     *
     * It does not include translation files, manifest
     * and signature.
     *
     * Final files list might differ due to export
     * conditions.
     */
    get files() {
        return Object.keys(this[filesSymbol]);
    }
    /**
     * Allows files to be added to the bundle.
     * If the bundle is closed, it will throw an error.
     *
     * @param fileName
     * @param buffer
     */
    addBuffer(fileName, buffer) {
        if (this.isFrozen) {
            throw new Error(Messages.BUNDLE.CLOSED);
        }
        this[filesSymbol][fileName] = buffer;
    }
    /**
     * Closes the bundle and returns it as a Buffer.
     * Once closed, the bundle does not allow files
     * to be added any further.
     *
     * @returns Buffer
     */
    getAsBuffer() {
        this[freezeSymbol]();
        return Buffer.from(zipToArray(createZipFilesMap(this[filesSymbol])));
    }
    /**
     * Closes the bundle and returns it as a stream.
     * Once closed, the bundle does not allow files
     * to be added any further.
     *
     * @returns
     */
    getAsStream() {
        this[freezeSymbol]();
        return Readable.from(Buffer.from(zipToArray(createZipFilesMap(this[filesSymbol]))));
    }
    /**
     * Closes the bundle and returns it as an object.
     * This allows developers to choose a different way
     * of serving, analyzing or zipping the file, outside the
     * default compression system.
     *
     * @returns a frozen object containing files paths as key
     * 		and Buffers as content.
     */
    getAsRaw() {
        this[freezeSymbol]();
        return Object.freeze({ ...this[filesSymbol] });
    }
}
/**
 * Creates a files map for do-not-zip
 *
 * @param files
 * @returns
 */
function createZipFilesMap(files) {
    return Object.entries(files).map(([path, data]) => ({
        path,
        data,
    }));
}
//# sourceMappingURL=data:application/json;base64,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