import * as path from "node:path";
import { promises as fs } from "node:fs";
import * as Utils from "./utils.js";
import * as Messages from "./messages.js";
/**
 * Reads the model folder contents
 *
 * @param model
 * @returns A promise of an object containing all
 * 		filePaths and the relative buffer
 */
export default async function getModelFolderContents(model) {
    try {
        const modelPath = `${model}${(!path.extname(model) && ".pass") || ""}`;
        const modelFilesList = await fs.readdir(modelPath);
        // No dot-starting files, manifest and signature and only files with an extension
        const modelSuitableRootPaths = Utils.removeHidden(modelFilesList).filter((f) => !/(manifest|signature)/i.test(f) &&
            /.+$/.test(path.parse(f).ext));
        const modelRecords = await Promise.all(modelSuitableRootPaths.map((fileOrDirectoryPath) => readFileOrDirectory(path.resolve(modelPath, fileOrDirectoryPath))));
        return Object.fromEntries(modelRecords.flat(1));
    }
    catch (err) {
        if (!isErrorErrNoException(err) || !isMissingFileError(err)) {
            throw err;
        }
        if (isFileReadingFailure(err)) {
            throw new Error(Messages.format(Messages.MODELS.FILE_NO_OPEN, JSON.stringify(err)));
        }
        if (isDirectoryReadingFailure(err)) {
            throw new Error(Messages.format(Messages.MODELS.DIR_NOT_FOUND, err.path));
        }
        throw err;
    }
}
function isErrorErrNoException(err) {
    return Object.prototype.hasOwnProperty.call(err, "errno");
}
function isMissingFileError(err) {
    return err.code === "ENOENT";
}
function isDirectoryReadingFailure(err) {
    return err.syscall === "scandir";
}
function isFileReadingFailure(err) {
    return err.syscall === "open";
}
/**
 * Allows reading both a whole directory or a set of
 * file in the same flow
 *
 * @param filePath
 * @returns
 */
async function readFileOrDirectory(filePath) {
    const stats = await fs.lstat(filePath);
    if (stats.isDirectory()) {
        return readFilesInDirectory(filePath);
    }
    else {
        return getFileContents(filePath).then((result) => [result]);
    }
}
/**
 * Reads a directory and returns all
 * the files in it
 *
 * @param filePath
 * @returns
 */
async function readFilesInDirectory(filePath) {
    const dirContent = await fs.readdir(filePath).then(Utils.removeHidden);
    return Promise.all(dirContent.map((fileName) => getFileContents(path.resolve(filePath, fileName), 2)));
}
/**
 * @param filePath
 * @param pathSlicesDepthFromEnd used to preserve localization lproj content
 * @returns
 */
async function getFileContents(filePath, pathSlicesDepthFromEnd = 1) {
    const fileComponents = filePath.split(path.sep);
    const fileName = fileComponents
        .slice(fileComponents.length - pathSlicesDepthFromEnd)
        .join("/");
    const content = await fs.readFile(filePath);
    return [fileName, content];
}
//# sourceMappingURL=data:application/json;base64,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